/*
 * Decompiled with CFR 0.152.
 */
package com.splunk.nscache;

import com.splunk.nscache.INode;
import com.splunk.nscache.INodeDirectory;
import com.splunk.nscache.INodeFile;
import com.splunk.nscache.INodeSymlink;
import com.splunk.nscache.UnresolvedPathException;
import com.splunk.util.StrUtil;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.WritableUtils;
import org.apache.log4j.Logger;

public class NSCache {
    private static final Logger gLogger = Logger.getLogger(NSCache.class);
    private File cacheFile;
    private boolean expired = true;
    private INodeDirectory root;
    private String vixName;
    private String inputId;
    private FileStatus rootStat;
    private long ttl;
    private long maxSize;
    private boolean compress;
    public static final long DEFAULT_CACHE_TTL = 28800000L;
    public static final long DEFAULT_CACHE_MAX_SIZE = 0x6400000L;
    public static final boolean DEFAULT_CACHE_COMPRESS = false;
    public static final String CACHE_NAME_DELIMITER = "_";

    public static File getCacheDir() {
        String splunkHome = System.getenv("SPLUNK_HOME");
        return new File(splunkHome, INode.constructPath("var", "run", "splunk", "hunk", "nscache"));
    }

    public static String getCacheFileNamePrefix(String vixName) {
        return MD5Hash.digest((String)vixName).toString();
    }

    public NSCache(String vixName, String inputId, FileStatus rootStat, long ttl, long maxSize, boolean compress) {
        this.vixName = vixName;
        this.inputId = inputId;
        this.rootStat = rootStat;
        this.ttl = ttl <= 0L ? 28800000L : ttl;
        this.maxSize = maxSize <= 0L ? 0x6400000L : maxSize;
        this.compress = compress;
        String fileName = NSCache.getCacheFileNamePrefix(vixName) + CACHE_NAME_DELIMITER + inputId + ".cache";
        if (compress) {
            fileName = fileName + ".gz";
        }
        this.cacheFile = new File(NSCache.getCacheDir(), fileName);
        if (!this.cacheFile.getParentFile().exists()) {
            this.cacheFile.getParentFile().mkdirs();
        }
    }

    public NSCache(String vixName, String inputId, FileStatus rootStat) {
        this(vixName, inputId, rootStat, 0L, 0L, false);
    }

    public FileStatus getRootFileStatus() {
        return this.rootStat;
    }

    private void cleanAbandonedCacheFile() {
        if (this.compress) {
            File uncompressed = new File(this.cacheFile.getParent(), this.cacheFile.getName().substring(0, this.cacheFile.getName().length() - ".gz".length()));
            if (uncompressed.delete()) {
                gLogger.info((Object)("Deleted an abandoned cache file: " + uncompressed.getPath()));
            }
        } else {
            File compressed = new File(this.cacheFile.getParent(), this.cacheFile.getName() + ".gz");
            if (compressed.delete()) {
                gLogger.info((Object)("Deleted an abandoned cache file: " + compressed.getPath()));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void load() throws IOException {
        this.cleanAbandonedCacheFile();
        if (!this.cacheFile.exists()) {
            this.expired = true;
            this.root = new INodeDirectory(null, this.rootStat);
            return;
        }
        InputStream in = new FileInputStream(this.cacheFile);
        if (this.compress) {
            in = new GZIPInputStream(in);
        }
        DataInputStream dis = new DataInputStream(new BufferedInputStream(in));
        try {
            long stime = System.currentTimeMillis();
            long grandTotal = 0L;
            long totalItems = dis.readLong();
            grandTotal += totalItems;
            gLogger.debug((Object)("totalItems in nscache: " + totalItems));
            this.root = (INodeDirectory)this.loadINode(dis, null);
            --totalItems;
            gLogger.debug((Object)("loaded root=" + this.root.getPath()));
            long lmtime = this.root.mtime;
            long rmtime = this.rootStat.getModificationTime();
            long cacheFileLife = System.currentTimeMillis() - this.cacheFile.lastModified();
            if (lmtime < rmtime || cacheFileLife > this.ttl) {
                this.expired = true;
                this.root = new INodeDirectory(null, this.rootStat);
                return;
            }
            byte[] bytes = new byte[]{};
            while (totalItems > 0L) {
                short len = dis.readShort();
                if (bytes.length < len) {
                    bytes = new byte[len];
                }
                dis.readFully(bytes, 0, len);
                String curPath = new String(bytes, 0, (int)len, StrUtil.UTF8);
                INodeDirectory current = (INodeDirectory)this.root.getNode(new Path(curPath));
                if (current == null) {
                    throw new UnresolvedPathException("Cannot find current node from the cache. current=" + curPath + ", root=" + this.root.getPath());
                }
                int childrenCount = dis.readInt();
                gLogger.debug((Object)("current=" + curPath + ", childrenCount=" + childrenCount));
                for (int j = 0; j < childrenCount; ++j) {
                    INode child = this.loadINode(dis, curPath);
                    current.addChild(child);
                }
                totalItems -= (long)childrenCount;
            }
            this.expired = false;
            gLogger.info((Object)("Load nscache took " + (System.currentTimeMillis() - stime) + " ms. totalItems=" + grandTotal + ", nscache=" + this.toString()));
        }
        finally {
            IOUtils.closeQuietly((InputStream)dis);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void save() throws IOException {
        if (!this.expired) {
            return;
        }
        long stime = System.currentTimeMillis();
        File tempCacheFile = new File(this.cacheFile.getAbsolutePath() + "." + UUID.randomUUID().toString() + ".tmp");
        OutputStream out = new FileOutputStream(tempCacheFile);
        if (this.compress) {
            out = new GZIPOutputStream(out);
        }
        DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(out));
        try {
            dos.writeLong(this.root.totalItems());
            gLogger.debug((Object)("saved totalItems=" + this.root.totalItems()));
            this.saveINode(dos, this.root);
            gLogger.debug((Object)("saved root=" + this.root.getPath()));
            byte[] bytes = new byte[32000];
            ByteBuffer curDirName = ByteBuffer.wrap(bytes);
            curDirName.put(StrUtil.string2Bytes(this.root.getPath().toString()));
            this.save(curDirName, this.root, dos);
            dos.close();
            if (tempCacheFile.length() > this.maxSize) {
                gLogger.warn((Object)("nscache file is too large, not saved. max_allowed_size=" + this.maxSize + ", cache_size=" + tempCacheFile.length() + ", nscache=" + tempCacheFile.getPath()));
                this.cacheFile.delete();
            } else {
                if (this.compress && !this.cacheFile.getName().endsWith(".gz")) {
                    File compressedFileName = new File(this.cacheFile.getAbsolutePath() + ".gz");
                    tempCacheFile.renameTo(compressedFileName);
                    this.cacheFile.delete();
                } else {
                    tempCacheFile.renameTo(this.cacheFile);
                }
                this.expired = false;
                gLogger.info((Object)("Save nscache took " + (System.currentTimeMillis() - stime) + " ms. totalItems=" + this.root.totalItems() + ", nscache=" + this.toString()));
            }
        }
        finally {
            IOUtils.closeQuietly((OutputStream)dos);
            tempCacheFile.delete();
        }
    }

    public boolean delete() {
        this.expired = true;
        long st = System.currentTimeMillis();
        boolean deleted = this.cacheFile.delete();
        gLogger.info((Object)("deleted nscache=" + this.toString() + " took " + (System.currentTimeMillis() - st) + " ms."));
        if (!deleted) {
            gLogger.warn((Object)("Failed to delete nscache. cache=" + this.toString()));
        }
        return deleted;
    }

    public boolean isExpired() {
        return this.expired;
    }

    INodeDirectory getRootNode() {
        return this.root;
    }

    public FileStatus getFileStatus(Path nodePath) throws UnresolvedPathException {
        INode node = this.root.getNode(nodePath);
        if (node == null) {
            throw new UnresolvedPathException("Cannot find the node. nodePath=" + nodePath);
        }
        return node.getFileStatus();
    }

    public void addNodes(FileStatus[] stats) throws IOException {
        if (stats == null) {
            return;
        }
        for (FileStatus stat : stats) {
            this.addNode(stat);
        }
    }

    public void addNode(FileStatus stat) throws IOException {
        this.root.addNode(stat);
    }

    public FileStatus[] listStatus(Path nodePath) throws UnresolvedPathException {
        ArrayList<FileStatus> stats = new ArrayList<FileStatus>();
        INode node = this.root.getNode(nodePath);
        if (node == null) {
            gLogger.warn((Object)("Cannot find the node. nodePath=" + nodePath + ", root=" + this.root.getPath()));
            return null;
        }
        if (node.getType() == INode.INodeType.DIR) {
            List<INode> children = ((INodeDirectory)node).getChildren();
            if (children != null) {
                for (INode n : children) {
                    stats.add(n.getFileStatus());
                }
            }
        } else {
            stats.add(node.getFileStatus());
        }
        return stats.toArray(new FileStatus[stats.size()]);
    }

    public String toString() {
        String rootPath = this.rootStat == null ? "" : this.rootStat.getPath().toUri().getPath();
        String cachePath = this.cacheFile == null ? "" : this.cacheFile.getAbsolutePath();
        return "[vixName=" + this.vixName + ", inputId=" + this.inputId + ", root=" + rootPath + ", cache=" + cachePath + "]";
    }

    private void save(ByteBuffer curDirName, INodeDirectory node, DataOutputStream out) throws IOException {
        List<INode> children = node.getChildren();
        if (children == null || children.isEmpty()) {
            return;
        }
        int prefixLen = curDirName.position();
        out.writeShort(prefixLen);
        out.write(curDirName.array(), 0, prefixLen);
        if (gLogger.isDebugEnabled()) {
            gLogger.debug((Object)("saved curDirPath=" + new String(curDirName.array(), 0, prefixLen)));
        }
        out.writeInt(children.size());
        gLogger.debug((Object)("saved children.size=" + children.size()));
        for (INode child : children) {
            this.saveINode(out, child);
        }
        for (INode child : children) {
            if (!child.isDir()) continue;
            curDirName.put(StrUtil.string2Bytes("/")).put(StrUtil.string2Bytes(child.getName()));
            this.save(curDirName, (INodeDirectory)child, out);
            curDirName.position(prefixLen);
        }
    }

    private void saveINode(DataOutputStream out, INode inode) throws IOException {
        if (out == null) {
            return;
        }
        out.writeByte(inode.getType().toValue());
        Text.writeString((DataOutput)out, (String)(inode.isRoot() ? inode.getPath().toString() : inode.getName()));
        WritableUtils.writeVInt((DataOutput)out, (int)inode.getReplication());
        WritableUtils.writeVLong((DataOutput)out, (long)inode.getBlockSize());
        out.writeLong(inode.mtime);
        out.writeLong(inode.atime);
        Text.writeString((DataOutput)out, (String)inode.getGroup());
        Text.writeString((DataOutput)out, (String)inode.getOwner());
        WritableUtils.writeVLong((DataOutput)out, (long)inode.length);
        out.writeShort(inode.permission);
        if (inode.getType() == INode.INodeType.SYMLINK) {
            Text.writeString((DataOutput)out, (String)((INodeSymlink)inode).getSymlink());
        }
        if (gLogger.isDebugEnabled()) {
            gLogger.debug((Object)("Saved current node info. curNode=" + inode.getName()));
        }
    }

    private INode loadINode(DataInputStream in, String parentPath) throws IOException {
        if (in == null) {
            return null;
        }
        INode.INodeType type = INode.INodeType.fromValue(in.readByte());
        String pathOrName = Text.readString((DataInput)in);
        short replication = (short)WritableUtils.readVInt((DataInput)in);
        long blockSize = WritableUtils.readVLong((DataInput)in);
        long mtime = in.readLong();
        long atime = in.readLong();
        String group = Text.readString((DataInput)in);
        String owner = Text.readString((DataInput)in);
        long length = WritableUtils.readVLong((DataInput)in);
        short permission = in.readShort();
        if (type == INode.INodeType.DIR) {
            return new INodeDirectory(parentPath, pathOrName, mtime, atime, group, owner, length, permission);
        }
        if (type == INode.INodeType.FILE) {
            return new INodeFile(parentPath, pathOrName, mtime, atime, group, owner, length, permission, replication, blockSize);
        }
        if (type == INode.INodeType.SYMLINK) {
            String symlink = Text.readString((DataInput)in);
            return new INodeSymlink(parentPath, pathOrName, mtime, atime, group, owner, length, permission, symlink);
        }
        throw new RuntimeException("Invalid type read from input stream: " + (Object)((Object)type));
    }
}

