/*
 * Decompiled with CFR 0.152.
 */
package com.splunk.roll;

import com.splunk.mr.cache.reporting.Report;
import com.splunk.roll.Bucket;
import com.splunk.roll.BucketExporter;
import com.splunk.roll.BucketMetadata;
import com.splunk.roll.PathResolver;
import com.splunk.roll.RollTransfer;
import com.splunk.roll.RollValidator;
import com.splunk.roll.Roller;
import com.splunk.roll.ThrottledIO;
import com.splunk.roll.util.ConfU;
import com.splunk.roll.util.OutputUtil;
import com.splunk.util.FunctionUtils;
import com.splunk.util.HdfsUtil;
import com.splunk.util.MapUtil;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.WeakHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.log4j.Logger;

public class Transactor {
    private static final String LOCAL_BUCKET_SIZE = "local_bucket_bytes";
    private static final String REMOTE_BUCKET_SIZE = "remote_bucket_bytes";
    private static final String REMOTE_TO_LOCAL_SIZE_RATIO = "remote_to_local_size_ratio";
    private static final String LATEST_BUCKET_TIME = "latest_bucket_time_secs";
    private static final String EARLIEST_BUCKET_TIME = "earliest_bucket_time_secs";
    private static final String SUCCESSFUL_BUCKET_TRANSACTION = "successful_bucket_transaction";
    private static final String ALL_TRANSACTIONS_SUCCEEDED = "all_transactions_succeeded";
    private static final String BUCKET_TRANSACTION_START = "bucket_transaction_start";
    private static final Logger gLogger = Logger.getLogger(Transactor.class);
    private static final String TEMP_FILENAME_PREFIX = Transactor.class.getName() + "-";
    private static final String TEMP_FILENAME_SUFFIX = ".tmp";
    private final PathResolver pathResolver;
    private final Configuration conf;
    private final File hunkRollTempDir;
    private final BucketExporter bucketExporter;
    private final BucketMetadata bucketMetadata;
    private final RollValidator rollValidator;
    private final WeakHashMap<Bucket, TempPaths> tempPaths;
    private final WeakHashMap<Bucket, Map<String, Object>> transactionMeta;

    public Transactor(Configuration conf, PathResolver pathResolver, File hunkRollTempDir, BucketExporter bucketExporter, BucketMetadata bucketMetadata, RollValidator rollValidator) {
        this.conf = conf;
        this.pathResolver = pathResolver;
        this.hunkRollTempDir = hunkRollTempDir;
        this.bucketExporter = bucketExporter;
        this.bucketMetadata = bucketMetadata;
        this.rollValidator = rollValidator;
        this.tempPaths = new WeakHashMap();
        this.transactionMeta = new WeakHashMap();
    }

    public void prepare(Bucket.LocalBucket bucket) throws IOException {
        this.transactionMeta.put(bucket, new HashMap());
        this.transactionMeta.get(bucket).put(BUCKET_TRANSACTION_START, System.currentTimeMillis());
        this.transactionMeta.get(bucket).put(SUCCESSFUL_BUCKET_TRANSACTION, false);
        File localPrepareBucketDir = this.resolveTempPaths((Bucket)bucket).local;
        Roller.ensureDirExistence(localPrepareBucketDir);
        Path remotePrepareDir = this.resolveTempPaths((Bucket)bucket).remote;
        FileSystem fs = remotePrepareDir.getFileSystem(this.conf);
        this.deleteHdfsPathQuietly(remotePrepareDir);
        fs.mkdirs(remotePrepareDir);
        this.transactionMeta.get(bucket).put(LOCAL_BUCKET_SIZE, FileUtils.sizeOfDirectory((File)bucket.getDir()));
        OutputUtil.writeMap(gLogger, "will prepare bucket transfer: ", MapUtil.asMap("bucket_name", bucket.getName(), "splunk_index", bucket.getIndex(), "directory", bucket.getDir().getAbsolutePath()));
        ArrayList<RollTransfer> transfers = new ArrayList<RollTransfer>();
        transfers.addAll(this.bucketExporter.export(bucket, localPrepareBucketDir, remotePrepareDir));
        transfers.addAll(this.bucketMetadata.addMeta(bucket, localPrepareBucketDir, remotePrepareDir));
        this.transactionMeta.get(bucket).put(REMOTE_BUCKET_SIZE, FunctionUtils.reduce(transfers, 0L, new FunctionUtils.RFn<RollTransfer, Long>(){

            @Override
            public Long apply(Long result, RollTransfer transfer) throws Exception {
                return result + transfer.size;
            }
        }));
        this.rollValidator.writeReceipt(remotePrepareDir, transfers);
    }

    public void commit(Bucket.LocalBucket bucket) throws IOException {
        Path realPath = this.pathResolver.resolvePath(bucket);
        Path tmpRemotePath = this.resolveTempPaths((Bucket)bucket).remote;
        FileSystem fs = realPath.getFileSystem(this.conf);
        fs.mkdirs(realPath.getParent());
        if (this.rollValidator.isRollValid(realPath, true)) {
            throw new AlreadyCopiedBucketException("Roll was already valid. Abort commit for bucket=" + bucket + " remote_path=" + realPath.makeQualified(fs));
        }
        this.deleteHdfsPathQuietly(realPath);
        boolean renamed = fs.rename(tmpRemotePath, realPath);
        if (!renamed) {
            String msg = "Commit was unsuccessful because the rename failed, ";
            if (!fs.exists(tmpRemotePath)) {
                msg = "Commit was unsuccessful because tmp file was missing. Possible file system consistency issue. ";
                throw new S3ConsistencyModelException(msg + "tmp_remote_path=" + tmpRemotePath.makeQualified(fs) + ", remote_path=" + realPath.makeQualified(fs));
            }
            throw new IOException(msg + "tmp_remote_path=" + tmpRemotePath.makeQualified(fs) + ", remote_path=" + realPath.makeQualified(fs));
        }
        Map<String, Object> meta = this.transactionMeta.get(bucket);
        meta.put(SUCCESSFUL_BUCKET_TRANSACTION, true);
        long transactionStart = (Long)meta.get(BUCKET_TRANSACTION_START);
        long transactionMs = System.currentTimeMillis() - transactionStart;
        long remoteSize = (Long)meta.get(REMOTE_BUCKET_SIZE);
        OutputUtil.writeMap(gLogger, "committed: ", MapUtil.asMap("bucket_name", bucket.getName(), "splunk_index", bucket.getIndex(), "directory", bucket.getDir().getAbsolutePath(), "remote_path", realPath.makeQualified(fs).toString(), "elapsed_ms", String.valueOf(transactionMs), LOCAL_BUCKET_SIZE, String.valueOf(meta.get(LOCAL_BUCKET_SIZE)), REMOTE_BUCKET_SIZE, String.valueOf(remoteSize)));
    }

    public void fillReport(Report report, Bucket.LocalBucket bucket) {
        Map<String, Object> meta = this.transactionMeta.get(bucket);
        report.earlyTime(EARLIEST_BUCKET_TIME, bucket.getTimeRange().getLow(TimeUnit.SECONDS));
        report.earlyTime(LATEST_BUCKET_TIME, bucket.getTimeRange().getHigh(TimeUnit.SECONDS));
        report.allTrue(ALL_TRANSACTIONS_SUCCEEDED, (Boolean)meta.get(SUCCESSFUL_BUCKET_TRANSACTION));
        this.reportBucketSizeData(report, meta);
        report.setRatio(REMOTE_TO_LOCAL_SIZE_RATIO, REMOTE_BUCKET_SIZE, LOCAL_BUCKET_SIZE);
    }

    private void reportBucketSizeData(Report report, Map<String, Object> meta) {
        if (meta.containsKey(SUCCESSFUL_BUCKET_TRANSACTION)) {
            if (meta.containsKey(LOCAL_BUCKET_SIZE)) {
                report.increment(LOCAL_BUCKET_SIZE, (Long)meta.get(LOCAL_BUCKET_SIZE));
            }
            if (meta.containsKey(REMOTE_BUCKET_SIZE)) {
                report.increment(REMOTE_BUCKET_SIZE, (Long)meta.get(REMOTE_BUCKET_SIZE));
            }
        }
    }

    public void clean(Bucket.LocalBucket bucket) {
        FileUtils.deleteQuietly((File)this.resolveTempPaths((Bucket)bucket).local);
        this.deleteHdfsPathQuietly(this.resolveTempPaths((Bucket)bucket).remote);
    }

    private void deleteHdfsPathQuietly(Path remotePath) {
        try {
            FileSystem fs = remotePath.getFileSystem(this.conf);
            Boolean exists = null;
            if (gLogger.isDebugEnabled()) {
                exists = fs.exists(remotePath);
            }
            boolean deleted = fs.delete(remotePath, true);
            if (gLogger.isDebugEnabled()) {
                gLogger.debug((Object)("Deleting remote path. existed=" + exists + " deleted=" + deleted));
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private TempPaths resolveTempPaths(Bucket bucket) {
        if (!this.tempPaths.containsKey(bucket)) {
            String uuid = UUID.randomUUID().toString();
            String tempDirName = Transactor.getNamespacedTempFileName(uuid);
            File local = new File(this.hunkRollTempDir, tempDirName);
            Path remote = new Path(this.getRemoteTempPath(), tempDirName);
            this.tempPaths.put(bucket, new TempPaths(local, remote));
        }
        return this.tempPaths.get(bucket);
    }

    private Path getRemoteTempPath() {
        return ConfU.getRollTemp(this.conf);
    }

    public Iterable<File> listLocalTempFiles() {
        File[] files = this.hunkRollTempDir.listFiles(this.getTempFilenameFilter());
        if (files == null) {
            return Collections.emptyList();
        }
        return Arrays.asList(files);
    }

    private FilenameFilter getTempFilenameFilter() {
        return new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return Transactor.isTempFile(filename);
            }
        };
    }

    public static String getNamespacedTempFileName(String tempFileNameBody) {
        return TEMP_FILENAME_PREFIX + tempFileNameBody + TEMP_FILENAME_SUFFIX;
    }

    public static boolean isTempFile(String name) {
        return name.startsWith(TEMP_FILENAME_PREFIX) && name.endsWith(TEMP_FILENAME_SUFFIX);
    }

    public Iterable<FileStatus> listRemoteTempFiles() throws IOException {
        return Transactor.listRemoteTempFilesIn(this.getRemoteTempPath(), this.conf);
    }

    public static List<FileStatus> listRemoteTempFilesIn(Path p, Configuration conf) throws IOException {
        return Arrays.asList(HdfsUtil.ls(p.getFileSystem(conf), p, new PathFilter(){

            public boolean accept(Path path) {
                return Transactor.isTempFile(path.getName());
            }
        }));
    }

    public static Transactor create(Configuration conf, PathResolver pathResolver, File hunkRollTempDir, RollValidator rollValidator, ThrottledIO throttledIO) {
        BucketExporter bucketExporter = BucketExporter.create(conf, throttledIO);
        BucketMetadata bucketMetadata = BucketMetadata.create(conf, throttledIO);
        return new Transactor(conf, pathResolver, hunkRollTempDir, bucketExporter, bucketMetadata, rollValidator);
    }

    public static class S3ConsistencyModelException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public S3ConsistencyModelException(String message) {
            super(message);
        }
    }

    public static class AlreadyCopiedBucketException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public AlreadyCopiedBucketException(String message) {
            super(message);
        }
    }

    private static class TempPaths {
        public final File local;
        public final Path remote;

        public TempPaths(File local, Path remote) {
            this.local = local;
            this.remote = remote;
        }
    }
}

