"""
Internationalization and Localization related stuff

We (should) only use unicode internally for Mr Sparkle

This module sets 3 variables on cherrypy.request for every request:
* cherrypy.request.lang - The user's primary locale that we're using (en_US for example)
* cherrypy.request.t - The GNUTranslations object providing message translations; you probably don't need to use this directly
* cherrypy.request.locale - A babel.Locale object that handles locale specific formating for dates, numbers, etc

You probably needn't access any of the above variables in normal use.

Use the _() function to translate simple unicode strings
use i18n.ungettext() to translate unicode strings with numbers in them
use i18n.datefmt() to return a babel.support.Format object that knows how to format dates
use i18.format_number() to format numbers for the current user's locale

TODO: I18NDispatcher should load the user's stored language setting in preference
to the browser supplied accept-language list
"""
from __future__ import absolute_import

from builtins import zip

from builtins import range
from builtins import object

import cherrypy
import babel
import babel.support
import babel.numbers
from babel.messages.pofile import unescape
import os
import os.path
import sys
import gettext
import copy
import shutil
import hashlib
from future.moves.urllib import parse as urllib_parse
from datetime import datetime, timedelta
import math
import splunk.appserver.mrsparkle.lib.util as util
import splunk.appserver.mrsparkle.lib.apps as apps
from splunk.util import format_local_tzoffset
import pickle
import json
import logging
import lxml.etree as et
import defusedxml.lxml as safe_lxml
import splunk.util
import re
import string
import time
import uuid

logger = logging.getLogger('splunk.appserver.lib.i18n')

ISO8609_MICROTIME='%Y-%m-%dT%H:%M:%S.%Q' + format_local_tzoffset()

LOCALE_PATH = os.path.join(os.path.dirname(__file__), '..', 'locale')
CACHE_PATH = util.make_absolute(os.path.join('var', 'run', 'splunk', 'appserver', 'i18n'))
INTERNAL_APPS = ['gettingstarted', 'launcher', 'learned', 'legacy', 'sample_app', 'search', 'splunkdeploymentmonitor', 'splunkforwarder', 'splunklightforwarder', 'user-prefs', 'splunk_datapreview', 'splunk_monitoring_console', 'dmc', 'splunk_instrumentation', 'splunk_gdi']


def current_lang(as_string=False): # moved to util
    return util.current_lang(as_string)

def parse_localestr(locale):
    return util.parse_localestr(locale)

def current_lang_url_component():  # moved to util
    return util.current_lang_url_component()

# Thread/request aware wrappers for the translation methods we're likely to use

def ugettext(message):
    """
    Translate a string message
    This method is also installed as "_" in builtins
    """
    if message == '':
        return message # avoid returning .mo system info
    try:
        if sys.version_info >= (3, 0):
            return cherrypy.request.t.gettext(message)
        else:
            return cherrypy.request.t.ugettext(message)
    except AttributeError:
        # Called outside the context of a cherrypy request
        return message

def ungettext(msgid1, msgid2, n):
    """
    Translate a string message with a number in it
    """
    try:
        if sys.version_info >= (3, 0):
            return cherrypy.request.t.ngettext(msgid1, msgid2, n)
        else:
            return cherrypy.request.t.ungettext(msgid1, msgid2, n)
    except AttributeError:
        return msgid1 if n==1 else msgid2

def deferred_ugettext(message):
    """
    Returns a proxy that translates a string each time it's accessed rather than when it's defined
    See the DeferCall() class for more details.
    Use this routine instead of calling DeferCall directly so that the message is extracted automatically
    by i18n-extract in the same way strings generated by _() are
    """
    return util.DeferCall(ugettext, message)

def deferred_ungettext(message):
    """
    See deferred_ugettext()
    """
    return util.DeferCall(ungettext, message)

def datefmt(tzinfo=None):
    """
    Format a date/time according to the user's locale
    """
    return babel.support.Format(cherrypy.request.locale, tzinfo=tzinfo)

# Convenient number format methods
# see babel.numbers for documentation on these methods
# calling these methods via this module means you can omit the locale parameter as it
# will be taken from the user's current settings
for f in ('get_currency_name', 'get_currency_symbol', 'get_decimal_symbol', 'get_plus_sign_symbol', 'get_minus_sign_symbol',
    'get_exponential_symbol', 'get_group_symbol', 'format_currency',
    'format_scientific', 'parse_number', 'parse_decimal'):
        code = """
def %(f)s(*a, **kw):
    kw['locale'] = cherrypy.request.locale
    return babel.numbers.%(f)s(*a, **kw)
""" % { 'f': f }
        exec(code)

# Convenient date/time format methods
# see babel.dates for documentation on these methods
for f in ('format_date', 'format_datetime', 'format_time'):
        code = """
def %(f)s(*a, **kw):
    kw['locale'] = cherrypy.request.locale
    return babel.dates.%(f)s(*a, **kw)
""" % { 'f': f }
        exec(code)

def format_decimal(number, format=None):
    result = babel.numbers.format_decimal(number, format=format, locale=cherrypy.request.locale)
    if cherrypy.request.lang == 'en_DEBUG':
        result = format_decimal.re.sub('0', result)
    return result
format_decimal.re = re.compile(r'\d')

format_number = format_decimal

def format_datetime_microseconds(dt, date_base_format='short', time_base_format='medium'):
    """
    Like format_datetime, but converts the seconds to seconds+microseconds as ss.QQQ
    Also lets you specify the format to use for date and time individually

    t must be either a datetime object or an ISO8601+microtime formatted string.
    """

    if not isinstance(dt, datetime):
        dt = splunk.util.parseISO(str(dt))

    d = babel.dates
    locale = cherrypy.request.locale

    try:
        time_format = d.get_time_format(locale=locale, format=time_base_format+'-microtime')
    except KeyError:
        time_format = d.get_time_format(locale=locale, format=time_base_format)
        time_format = time_format.pattern if isinstance(time_format, d.DateTimePattern) else time_format
        time_format = time_format.replace('ss', 'ss_TTT') # seconds.microseconds
        time_format = locale.time_formats[time_base_format+'-microtime'] = d.parse_pattern(time_format)

    return d.get_datetime_format(time_base_format, locale=locale) \
        .replace('{0}', d.format_time(dt, time_format, tzinfo=None,
                                    locale=locale)) \
        .replace('{1}', d.format_date(dt, date_base_format, locale=locale))

def locale_uses_12h():
    """
    Returns True if the current locale displays times using the 12h clock
    """
    time_format = babel.dates.get_time_format(locale=cherrypy.request.locale, format='medium')
    return '%(a)' in time_format.format

def format_percent(number, format=None):
    """
    Display up to 3 decimal places for the appserver
    """
    locale = cherrypy.request.locale
    if not format:
        format = locale.percent_formats.get(None)
        format.frac_prec = (0, 3) # display between 0 and 3 digits of decimal precision
    return babel.numbers.format_percent(number, format=format, locale=locale)


def format_bytes(number, mantissa=2, base=2, output_units=None, split_return=False, use_iec=False, format=None):
    '''
    Returns an auto-scaled storage number based on human-readable format.

    PARAMS

        number: A positive number that represents a storage amount in bytes
        base: {2 | 10} - the base in which to convert units;
            2 -- 1KB = 1024B
            10 -- 1kB == 1000B
        output_units: {B | KB | MB | ... | YB} - a static unit in which to
            output the storage value. Set to None for auto scaling.
        split_return: {True | False} - Indicates if the return value is
            a single string or a tuple of value and unit
        use_iec: {True | False} - Indicates if the unit suffix should
            render in IEC format
        format: The i18n format specifier

    OUTPUT

        split_return = False (default)
            a localized string

        split_return = True
            a tuple that contains the converted number and a translated unit
            specifier.
    '''

    if number == None:
        raise ValueError('number cannot be None')

    if base == 2:
        table_index = 0
        if use_iec:
            display_index = 2
        else:
            display_index = 3
    elif base == 10:
        table_index = 1
        display_index = 3
    else:
        raise ValueError('base must be either 2 or 10')

    # define the mapping from value magnitude to friendly suffix
    prefix_table = [
        [80, 24, 'YiB', 'YB'],
        [70, 21, 'ZiB', 'ZB'],
        [60, 18, 'EiB', 'EB'],
        [50, 15, 'PiB', 'PB'],
        [40, 12, 'TiB', 'TB'],
        [30,  9, 'GiB', 'GB'],
        [20,  6, 'MiB', 'MB'],
        [10,  3, 'KiB', 'KB'],
        [0,   0,   'B',  'B'],
    ]

    if number == 0:
        magnitude = 1
    elif base == 10:
        magnitude = math.floor(math.log10(abs(number)))
    else:
        magnitude = math.floor(math.log(abs(number), base))
    for row in prefix_table:
        if (output_units != None and (output_units in row[2:4])) \
            or (output_units == None and magnitude >= row[table_index]):
            adjusted_value = round(float(abs(number)) / math.pow(base, row[table_index]), mantissa)
            if number < 0:
                adjusted_value *= -1

            if split_return:
                return {
                    'value': format_decimal(adjusted_value, format),
                    'unit': row[display_index]
                }
            else:
                return '%s %s' % (format_decimal(adjusted_value, format), row[display_index])

    raise ValueError('unable to format: %s' % number)



def format_timedelta(time1, time2=None):
    '''
    Returns a friendly relative time phrase by describing the time difference
    of time2 - time1 in human readable units.  time2 defaults to now()
    '''

    if isinstance(time1, int) or isinstance(time1, float):
        time2 = time2 or time.time()
        diff = time2 - time1
        delta = timedelta(seconds=diff)
    elif isinstance(time1, datetime):
        time2 = time2 or datetime.now(splunk.util.localTZ)
        delta = time2 - time1
    elif isinstance(time1, time.struct_time):
        time2 = time2 or time.gmtime()
        delta = timedelta(seconds=(time.mktime(time2) - time.mktime(time1)))
    else:
        logger.warn('cannot parse invalid time: %s' % time1)
        return 'n/a'

    if delta.days >= 30:
        months = delta.days // 30
        return ungettext("%(count)s month ago", "%(count)s months ago", months) % {'count': months}
    elif delta.days >= 7:
        weeks = delta.days // 7
        return ungettext("%(count)s week ago", "%(count)s weeks ago", weeks) % {'count': weeks}
    elif delta.days >= 1:
        return ungettext("%(count)s day ago", "%(count)s days ago", delta.days) % {'count': delta.days}
    elif delta.seconds >= 3600:
        hours = delta.seconds // 3600
        return ungettext("%(count)s hour ago", "%(count)s hours ago", hours) % {'count': hours}
    elif delta.seconds >= 60:
        minutes = delta.seconds // 60
        return ungettext("%(count)s minute ago", "%(count)s minutes ago", minutes) % {'count': minutes}
    elif delta.seconds > 0:
        return ungettext("%(count)s second ago", "%(count)s seconds ago", delta.seconds) % {'count': delta.seconds}
    else:
        return _('just now')



def make_i18n_url(path, translate=True):  # move to util
    """
    Translate a request path into an i18n path by prefixing the user's
    current locale onto the url: /en-US/account/login
    If translate==True true then also prefixes the configured root_endpoint to the path

    You probably don't want to call this anyway, you probably want make_url() or make_route()
    """
    path = path.strip('/')
    locale = current_lang()
    # return "fr-FR" if the locale component is available, else just return "fr"
    locale = "%s-%s" % (locale[0], locale[1]) if locale[1] else locale[0]
    root_endpoint = cherrypy.request.config.get('root_endpoint')
    localed_path = "/%s/%s" % (locale, path)

    if translate and root_endpoint not in ('/', '', None):
        return '%s/%s/%s' % (root_endpoint, locale, path)
    return localed_path

def strip_i18n_url(path):  # move to util
    """
    Return a URL path stripped of the root_endpoint prefix and the en-US segment
    """
    return '/' + path[len(cherrypy.request.script_name)+7:]

def path_to_i18n_paths(path):
    """
    Returns a list of possible filenames based on the active user's current language/locale choice
    from most specific to least specific
    For example, if the user is requested /static/foo.jpg and the user's
    current locale is fr_FR then returns
    [ '/static/foo-fr_FR.jpg', '/static/foo-fr.jpg', '/static/foo.jpg' ]
    """
    # construct possible filenames based on current lang/locale
    lang, loc, enc = current_lang()
    fnlist = []
    fn, ext = os.path.splitext(path)
    if loc:
        fnlist.append('%s-%s_%s%s' % (fn, lang, loc, ext))
    if lang:
        fnlist.append('%s-%s%s' % (fn, lang, ext))
    fnlist.append(path)
    return fnlist


class I18NDispatcher(cherrypy.dispatch.Dispatcher):
    """
    Customized request dispatcher
    This dispatcher strips the locale prefix from the URL (/fr-FR/...) before searching
    for a page handler
    request.path_info is still left with the original url however
    """
    # regex to check locale contains valid characters
    checkre = re.compile(r'^[\w-]+$')

    def __call__(self, path_info):
        request = cherrypy.request

        request.config = cherrypy.config.copy()
        path = path_info.strip('/').split('/') # strip the root prefix
        if not (path and path[0]):

            # SPL-79993 - splunkweb behind reverse proxy does not handle "HTTP redirect" correctly
            if request.headers.get('X-Forwarded-Proto', None) == 'https' or request.headers.get('X-Forwarded-Ssl', None) == 'on':
                base = request.base
                request.base = base.replace('http://', 'https://')

            # front page, get the user's language choice and redirect
            locale = self._set_default_locale()
            # redirect to the i18n'd front page url
            request.handler =  cherrypy.HTTPRedirect(make_i18n_url('/'))
            return

        # strip the /fr-FR/ locale component
        locale = path[0].replace('-', '_')
        if locale == 'en_DEBUG':
            logger.warn('USING DEBUG TRANSLATIONS')
            request.lang, request.mofile, request.t = get_translation('messages', ['en_US'])
            request.lang = 'en_DEBUG'
            request.t = SparkleDebugTranslations(open(request.mofile, 'rb'))
            request.locale = DebugLocale('en', 'US')
            request.locale_name = locale
        else:
            if self.checkre.match(locale):
                # check that locale is supported
                request.lang, request.mofile, request.t = get_translation('messages', [locale])
            else:
                request.lang, request.mofile, request.t = (None, None, None)
            if not cherrypy.request.lang:
                # see if the locale component actually looks like a locale, if not assume the user
                # has just omitted it and redirect according to their language preference
                if len(locale)==2 or (len(locale)>3 and locale[2]=='_'):
                     # LIGHT-2338: set the default locale to render properly the 'fancy' error page
                     locale = self._set_default_locale()
                     request.handler = cherrypy.HTTPError(404, "Invalid language specified")
                     return
                locale = self._set_default_locale()
                path_info = urllib_parse.quote(path_info)
                target = "%s?%s" % (path_info, urllib_parse.quote(request.query_string, '&%=+#')) if len(request.query_string) else path_info
                request.handler =  cherrypy.HTTPRedirect(make_i18n_url(target))
                return

            # we use the user's first choice of locale, even if a translation isn't available
            try:
                request.locale = babel.Locale.parse(locale)
            except babel.UnknownLocaleError:
                 request.handler = cherrypy.HTTPError(404, "Invalid language specified")
                 return
            request.locale_name = locale

        # commit the modified request path
        path_info = '/'.join(path[1:]) + ('/' if path_info[-1] == '/' else '')
        # use the default dispatcher logic once the path has been modified to remove the locale component
        return super(I18NDispatcher, self).__call__(path_info)

    def _set_default_locale(self):
        """
        Set the locale to the user's default preference
        """
        # XXX this needs to load the user's language preference here and override accept-language
        request = cherrypy.request
        locales = parse_accept_languages(request.headers.get('accept-language', 'en-US'))
        request.lang, request.mofile, request.t = get_translation('messages', locales)

class SparkleTranslations(gettext.GNUTranslations):
    def getCatalogEntries(self, msgid):
        """
        Fetch catalog entries for a given msgid, traversing the fallback tree if required
        Returns a single string if the msgid doesn't match a plural string
        returns a list of strings otherwise, from 0 through to n
        """
        if msgid in self._catalog:
            return self._catalog[msgid]
        elif (msgid, 0) in self._catalog:
            return [ self._catalog[(msgid, n)] for n in range(50) if (msgid, n) in self._catalog ]
        elif self._fallback:
            return self._fallback.getCatalogEntries(msgid)
        return None

    def getAllCatalogItems(self):
        """
        return a list of ALL catalog entries for this locale
        recursively include all fallback catalogs
        """
        # need to make sure that items is a different object than self._catalog since
        #  we will be calling extend with the fallback catalogs
        items = []
        items.extend(self._catalog.items())
        if self._fallback:
            items.extend(self._fallback.getAllCatalogItems())
        return items

class SparkleDebugTranslations(gettext.GNUTranslations):
    """
    Debug translations switch all characters of the text to a unicode marker character
    Variable subtituions (%s or %(foo)s) are maintained as as html markup tags
    """
    re = re.compile(r'(\%(:?\(\w+\))?\w)|(<[^>]+>)|(\s+)')
    ch = u'\u270c'

    def _trstr(self, msg):
        parts =  self.re.split(msg)
        result = []
        for i, part in enumerate(parts):
            if not part or (i and parts[i-1] and parts[i-1][:2]=='%('):
                continue
            if part[0] == '%':
                result.append('**'+part+'**')
            elif part[0] == '<' or part[0] in string.whitespace:
                result.append(part)
            else:
                result.append(self.ch * len(part))
        return u''.join(result)

    def getCatalogEntries(self, msgid):
        # foo %s bar %(thing)s
        return self._trstr(msgid)

    def getAllCatalogItems(self):
        return []

    def ugettext(self, message):
        return self._trstr(message)

    def ungettext(self, msgid1, msgid2, n):
        return self._trstr(msgid1)

def get_catalog_entries(msgid):
    return cherrypy.request.t.getCatalogEntries(msgid)

def get_all_catalog_items():
    return cherrypy.request.t.getAllCatalogItems()

_translations = {}
_translations_by_locale = {}
_no_translation = object()
def get_translation(domain, languages):
    """
    Find and fetch a translation catalog for a given domain and language set
    Analagous to the gettext.translation function, but also returns the matched locale name
    returns (locale name, catalog)
    """
    nelangs = []
    primary = None
    primary_mo = None
    t = None
    result = None
    for lang in languages:
        for nelang in gettext._expand_lang(lang):
            if nelang not in nelangs:
                nelangs.append(nelang)
    for lang in nelangs:
        mofiles = [os.path.join(LOCALE_PATH, lang, 'LC_MESSAGES', '%s.mo' % domain)]
        # add translations defined by applications, if available
        for appname, app in apps.local_apps.items(): # local_apps is not an instance of dictionary
            app_mofile = os.path.join(app['full_path'], 'locale', lang, 'LC_MESSAGES', '%s.mo' % domain)
            mofiles.append(app_mofile)
        import cherrypy
        for mofile in mofiles:
            key = os.path.abspath(mofile)
            t = _translations.get(key)
            if t is _no_translation:
                continue
            if not os.path.exists(mofile):
                _translations[key] = _no_translation
                continue
            if not primary:
                primary = lang
                primary_mo = mofile
            if t is None:
                t = _translations.setdefault(key, SparkleTranslations(open(mofile, 'rb'))) # was GNUTranslations
                _translations_by_locale.setdefault(lang, t)
            # Copy the translation object to allow setting fallbacks and
            # output charset. All other instance data is shared with the
            # cached object.
            t = copy.copy(t)
            if result is None:
                result = t
            else:
                result.add_fallback(t)
    return (primary, primary_mo, result)


def parse_accept_languages(langlist):
    """
    langlist should be an rfc2616 language list: en-us,en;q=0.7,fr-fr;q=0.3
    returns a list of language_LOCALE strings ordered highest priority first
    """
    langs = []
    default_fallback = 'en-US;q=0.0'
    seen = set()
    for entry in langlist.split(',') + [default_fallback]:
        langpri = entry.strip().split(';')
        if len(langpri)>1:
            entry = langpri[0].strip()
            priority = float(langpri[1][langpri[1].index('q=')+2:].strip())
        else:
            priority = 1
        ll = entry.split('-')
        lang, locale = ll[0].lower(), ll[1].upper() if len(ll)>1 else ''
        if not (lang, locale) in seen:
            langs.append( (priority, lang, locale) )
            seen.add((lang, locale))
    langs.sort(reverse=True)
    return [ "%s_%s" % (lang, locale) if locale else lang for priority, lang, locale in langs ]


def c2js(plural):
    """
    Gets a C expression as used in PO files for plural forms and returns a
    Javascript function that implements an equivalent expression.
    This is much simpler than the c2py function from gettext as the syntax between
    C and JS is basically the same for the algorithms likely to be implemented in a .po file

    We forgo the security checks too as we trust our translators
    """

    return "function(n) { return %s; }" % plural



def find_trans_keys_for_file_direct(filename):
    """
    Lookup a list of strings to be translated for a given file
    This relies on the messages.pot file being correctly marked up with
    the name of the source file associated with each string to be translated
    """
    filename = os.path.realpath(filename).lower()
    app_prefix = util.get_apps_dir().lower()
    app_name = None
    if filename.startswith(app_prefix):
        # if this is for a file defined in an app, then check it's local messages.pot file
        app_name =filename[len(app_prefix):].split(os.path.sep, 2)[1]
        potfile = os.path.join(app_prefix, app_name, 'locale', 'messages.pot')
        if app_name in INTERNAL_APPS:
            potfile = os.path.join(LOCALE_PATH, 'messages.pot')
    else:
        # else use the system messages.pot
        potfile = os.path.join(LOCALE_PATH, 'messages.pot')
    result = []
    try:
        f = open(potfile, 'r')
    except IOError:
        if app_name:
            logger.debug("Application %s does not contain a messages.pot file" % app_name)
            return result
        else:
            raise # Can't find the appserver messages.pot file == fatal.

    matched = False
    msgid = msgid_plural = None
    for line in f:
        line = line.strip()
        if line[:2] == '#:':
            fn = line[2:].rsplit(':', 1)[0].strip() # split off the line number
            try:
                # strip the relative portion of the filename
                fn = fn[fn.rindex('../')+3:]
            except ValueError:
                pass
            # hack for search_mrsparkle directories
            if fn.startswith('web/'):
                fn = fn[4:]
            if msgid:
                # process the existing entry for the last match
                result.append( (msgid, msgid_plural) )
                msgid = msgid_plural = None
                matched = False
            if not matched:
                # multiple files may use this string; we only need to know if 1 matched
                matched = fn.lower() == filename[-len(fn):].replace(os.path.sep, "/")
        elif matched:
            if line.startswith('msgid '):
                msgid = unescape(line[6:])
            elif line.startswith('msgid_plural '):
                msgid_plural = unescape(line[13:])
            elif line.startswith('"'):
                if msgid is not None:
                    msgid = msgid + unescape(line)
                elif msgid_plural is not None:
                    msgid_plural = msgid_plural + unescape(line)
    if msgid:
        result.append( (msgid, msgid_plural) )
    f.close()

    return result

_js_trans_keys = {}
def find_trans_keys_for_file(filename):
    """
    Lookup strings to be translated for a given file.
    This looks for a cache pickle file first and falls back to the
    slower method of crawling the .pot file if not found
    """
    global _js_trans_keys

    filename = os.path.realpath(filename).lower()
    app_prefix = util.get_apps_dir().lower()
    app_name = None
    orgfilename = filename
    if filename.startswith(app_prefix):
        # if this is for a file defined in an app, then check its local cache file
        app_name = filename[len(app_prefix):].split(os.path.sep, 2)[1]
        cachefile = os.path.join(app_prefix, app_name, 'locale', 'messages-filecache.bin')
        filename = filename[len(app_prefix)+1:]
        if app_name in INTERNAL_APPS:
            cachefile = os.path.join(LOCALE_PATH, 'messages-filecache.bin')
    else:
        # else use the system cache file
        cachefile = os.path.join(LOCALE_PATH, 'messages-filecache.bin')
        offset = filename.find('search_mrsparkle')
        if offset >= 0:
            filename = filename[offset:]
        else:
            filename = filename.split(os.sep)[-1]
    result = []
    filename = filename.replace(os.path.sep, '/')
    if cachefile in _js_trans_keys:
        data = _js_trans_keys[cachefile]
    else:
        f = None
        try:
            f = open(cachefile, 'rb')
            if sys.version_info >= (3, 0):
                data = pickle.load(f, encoding="bytes")
            else:
                data = pickle.load(f)
        except IOError:
            # fallback to the older/slower scan
            logger.debug("Failed to find i18n cache file for %s" % (orgfilename,))
            return find_trans_keys_for_file_direct(orgfilename)
        finally:
            if f:
                f.close()
        _js_trans_keys[cachefile] = data

    if data:
        return data.get(filename, [])
    return []

def get_plural_rule(locale=None):
    """
    returns a string representing the plural rule for the specified locale
    or, if a locale is not specified, for the current locale
    """

    if not locale:
        locale = cherrypy.request.lang
    catalog = _translations_by_locale.get(locale)
    if not catalog:
        r = get_translation('messages', [locale])
        catalog = _translations_by_locale.get(locale)
        if not catalog:
            return None

    if 'plural-forms' in catalog._info:
        # fetch the c syntax detailing how plurals should map to translations
        plural = catalog._info['plural-forms'].split(';')[1]
        plural = plural.split('=', 1)[1]
    else:
        plural = 'n==1 ? 0 : 1'

    return plural

def get_file_translations(filename, locale=None):
    """
    Fetch all static translations defined for a given file and locale
    """
    result = {'plural': get_plural_rule(locale), 'catalog': {}}

    keys = find_trans_keys_for_file(filename)
    for (msgid, msgid_plural) in keys:
        entries = get_catalog_entries(msgid)
        if not entries:
            result['catalog'][msgid] = msgid
            continue
        if isinstance(entries, list):
            for n, entry in enumerate(entries):
                result['catalog'][(msgid, n)] = entry
        else:
            result['catalog'][msgid] = entries

    return result

def javascript_stringify_catalog(catalog):
    # swap out the plural function with something we can replace
    plural_function = catalog['plural']
    plural_placeholder = "____REPLACE_ME_FUNCTION____"

    if plural_function:
        catalog['plural'] = plural_placeholder
        json_output = json.dumps(catalog)
        json_output = json_output.replace('"' + plural_placeholder + '"', plural_function)
    else:
        json_output = json.dumps(catalog)

    return json_output

def get_all_translations(locale=None):
    """
    Fetch all translations defined for a given locale
    """
    result = {'catalog': {}}
    result['plural'] = c2js(get_plural_rule(locale))

    catalogItems = get_all_catalog_items()
    result['catalog'] = dict([ ("%s-%s" % (msgid[1], msgid[0]), val) if isinstance(msgid, tuple) else ("+-%s" % (msgid,), val) for (msgid, val) in catalogItems ])
    return result

def get_all_translations_cached(locale=None, autoload=False):
    """
    Fetch all translations defined for a given locale. First try from cache, if not in cache, then build and store in cache.
    If autoload, then cache and return executable javascript string: "i18n_register(....)"
    If !autoload, then cache and return JSON object
    """
    autoload_specifier = "autoload" if autoload else "no-autoload"
    if not locale:
        locale = cherrypy.request.lang
    cache_file = os.path.join(CACHE_PATH, "catalog-%s-%s.cache" % (autoload_specifier, locale))

    # check to see if cache file exists, if so, return data therein
    if _check_i18n_cache(None, cache_file, locale):
        f = open(cache_file)
        output = f.read()
        return output

    logger.info("No cache file exists for entire catalog for locale=%s. Generating catalog" % locale)

    # else, generate the data
    catalog_contents = get_all_translations()
    if autoload:
        output = 'i18n_register(%s)' % javascript_stringify_catalog(catalog_contents)
    else:
        output = javascript_stringify_catalog(catalog_contents)

    # cache data, first into a tmp_file, and then atomic move to cache_file
    tmp_file = "%s-%s" % (cache_file, uuid.uuid4())
    try:
        f = open(tmp_file, "w")
        f.write(output)
        f.close()
        os.rename(tmp_file, cache_file)
        logger.info("Cache file generated for entire catalog for locale=%s" % locale)
    except Exception as e:
        logger.warning("Could not write cache file for entire locale catalog. error=%s" % e)
    finally:
        # make sure we remove the tmp_file if we couldn't rename it
        try:
            os.remove(tmp_file)
        except:
            pass

    return output

def init_i18n_cache(flush_files=False):
    # Check to see if the build number has changed since splunkweb was last started
    # if so, delete all the cache data for js translation information
    build_number = str(cherrypy.config['build_number'])
    if not os.path.exists(CACHE_PATH):
        os.makedirs(CACHE_PATH)
    try:
        f = open(os.path.join(CACHE_PATH, 'version'), 'r')
        current = str(f.read())
        f.close()
    except (IOError, ValueError):
        current = '0'
    # we want to delete all files when starting splunk
    # so if flush_files is set or if build no is different then rebuild the cache
    if flush_files or build_number != current:
        logger.info('Build number has changed to "%s"; flushing i18n cache' % build_number)
        for fn in os.listdir(CACHE_PATH):
            os.unlink(os.path.join(CACHE_PATH, fn))
        f = open(os.path.join(CACHE_PATH, 'version'), 'w')
        f.write(str(build_number))
        f.close()

def _check_i18n_cache(filename, cache_file, locale):
    if not os.path.exists(CACHE_PATH):
        os.makedirs(CACHE_PATH)
        return None  # if CACHE_PATH didn't exist, then cache_file doesn't either
    if os.path.exists(cache_file):
        if os.path.getsize(cache_file) == 0:
            return False # we've already checked and there's no translations for this file
        cache_mtime = os.path.getmtime(cache_file)

        if cache_mtime > os.path.getmtime(cherrypy.request.mofile) and (filename==None or cache_mtime > os.path.getmtime(filename)):
            return cache_file
    return None

def translate_js(filename, locale=None):
    if not locale:
        locale = cherrypy.request.lang
    path = os.path.abspath(filename)
    if sys.version_info >= (3, 0): path = path.encode()
    cache_file = os.path.join(CACHE_PATH, "%s-%s-%s.cache" % (os.path.basename(filename), hashlib.sha1(path).hexdigest(), locale))
    r = _check_i18n_cache(filename, cache_file, locale)
    if r is not None:
        return r
    tmp_file = cache_file + '.' + str(uuid.uuid4()) + '.cache'
    json_table = generate_js(filename, locale)
    if not json_table:
        # create an empty marker cache file
        f = open(cache_file, 'w')
        f.close()
        return False
    # SPL-189148: Explicitly specify utf-8, as sometimes this was opened as something else (i.e. latin-1)
    with open(tmp_file, 'w', newline='\n', encoding='utf-8') as f:
        f.write("\n// Translations for %s\n" % locale)
        f.write("i18n_register("+json_table+");\n\n\n")

        with open(filename, 'rb') as filename_fp:
            filename_contents_raw = filename_fp.read()

        try:
            filename_contents = filename_contents_raw.decode('utf-8')
        except UnicodeDecodeError:
            # Use chardetect to try to detect the encoding, and convert to
            # Unicode. This can be slow.
            logger.warn("%s is not encoded in ASCII or UTF-8. Please use one of these encodings." % filename)
            import chardet
            detected_encoding = chardet.detect(filename_contents_raw)
            filename_contents = filename_contents_raw.decode(detected_encoding['encoding'])

        f.write(filename_contents)

    try:
        if os.path.exists(cache_file):
            # SPL-87795 for some reason we tried to rename over existing cache_file here
            os.remove(cache_file)
        os.rename(tmp_file, cache_file)
    except:
        logger.warn('could not rename %s to %s' % (tmp_file, cache_file))
        return tmp_file
    return cache_file

def generate_js(filename, locale=None):
    """Generate the JS block that gets prepended to the .js file.  Also used to help generate the master modules.js file"""
    if not locale:
        locale = cherrypy.request.lang

    trans_table = get_file_translations(filename, locale)
    if not trans_table:
        return False

    trans_table['plural'] = c2js(trans_table['plural'])
    # Javascript can't use tuples as keys, so flatten them to strings
    trans_table['catalog'] = dict([ ("%s-%s" % (msgid[1], msgid[0]), val) if isinstance(msgid, tuple) else ("+-%s" % (msgid,), val)  for msgid, val in trans_table['catalog'].items() ])
    return javascript_stringify_catalog(trans_table)

def generate_wrapped_js(filename, locale=None):
    """This is so outside resources (filechain, etc.) can retrieve the i18n JS block without writing to a file"""
    js = generate_js(filename, locale)
    if js:
        return 'i18n_register(%s);' % js
    else:
        return False

def dispatch_i18n_js(filename):
    """Send i18n.js and add user's locale data to it"""
    # cherrypy.request does not implement get()
    try:
        locale_name = cherrypy.request.locale_name
    except:
        locale_name = 'en-US'

    path = os.path.abspath(filename)
    if sys.version_info >= (3, 0): path = path.encode()
    cache_file = os.path.join(CACHE_PATH, "%s-%s-%s.cache" % (os.path.basename(filename), hashlib.sha1(path).hexdigest(), locale_name))
    r = _check_i18n_cache(filename, cache_file, locale_name)
    if r is not None:
        return r

    try:
        locale = cherrypy.request.locale
    except:
        locale = babel.Locale.parse('en')

    def parse(data):
        result = {}
        for key, val in data.items():
            if isinstance(val, babel.localedata.LocaleDataDict):
                result[key] = parse(val)
            else:
                result[key] = val
        return result

    data = {
        'locale_name': locale_name,
        'minus_sign': locale.number_symbols.get('minusSign', u'-'),
        'plus_sign': locale.number_symbols.get('plusSign', u'+'),
        'exp_symbol': locale.number_symbols.get('exponential', u'E'),
        'decimal_symbol': locale.number_symbols.get('decimal', u'.'),
        'group_symbol': locale.number_symbols.get('group', u','),
        'number_format': locale.decimal_formats.get(None).pattern,
        'percent_format': locale.percent_formats.get(None).pattern,
        'scientific_format': locale.scientific_formats.get(None).pattern,
        'periods': parse(locale.periods),
        'days': parse(locale.days),
        'months': parse(locale.months),
        'quarters': parse(locale.quarters),
        'eras': parse(locale.eras),
        'date_formats': dict( [ (key, { 'format': val.format, 'pattern': val.pattern }) for key, val in locale.date_formats.items() ]),
        'time_formats': dict( [ (key, { 'format': val.format, 'pattern': val.pattern }) for key, val in locale.time_formats.items() ]),
        'first_week_day': locale.first_week_day,
        'min_week_days': locale.min_week_days,
    }

    # Restore previous Babel behavior for datetime rendering. All locales we currently
    # support follow the "date then time" pattern.
    data['datetime_formats'] = {'null': '{1} {0}'}

    json_data = json.dumps(data)
    f = open(cache_file, 'w', newline='\n')
    f.write("\n// Locale data for %s\n" % locale_name)
    f.write("_i18n_locale="+str(json_data)+";\n")
    write_moment_translation(find_js_folder(filename), str(locale), f)
    write_numeral_translation(find_js_folder(filename), str(locale), f)
    write_jquery_ui_datepicker_translation(find_js_folder(filename), str(locale), f)
    shutil.copyfileobj(open(filename, 'r'), f)
    # adding the translations on the bottom so i18n_register exists
    #TODO: split out the time formatters so this is no needed.
    f.write("\n// Translations for %s\n" % locale_name)

    wrapped_js = generate_wrapped_js(filename)
    if wrapped_js:
       f.write(wrapped_js)

    f.close()
    return cache_file

def find_js_folder(filename):
    base = None
    while base != "js":
        dirname, base = os.path.split(filename)
        filename = dirname
    return os.path.join(filename, base)

def write_moment_translation(js_path, locale, dest_file):
    lang_parts = locale.lower().split('_')
    while len(lang_parts):
        lang = "-".join(lang_parts)
        lang_path = os.path.join(js_path, 'contrib', 'moment', 'lang', '%s.js' % lang)
        if os.path.exists(lang_path):
            dest_file.write('\nfunction moment_install(moment) {\n')
            dest_file.write('\t// moment.js API backwards compatibility\n')
            dest_file.write('\tif(!moment.locale) { moment.locale = moment.lang; }\n')
            dest_file.write('\tif(!moment.defineLocale) { moment.defineLocale = moment.locale; }\n')
            src = open(lang_path, 'r', 128*1024)
            dest_file.write(src.read().replace("""require('../moment').""", "moment.", 1))
            src.close()
            dest_file.write('\n}\n')
            return
        del lang_parts[-1]

def write_numeral_translation(js_path, locale, dest_file):
    lang_parts = locale.lower().split('_')
    while len(lang_parts):
        lang = "-".join(lang_parts)
        lang_path = os.path.join(js_path, 'contrib', 'numeral', 'lang', '%s.js' % lang)
        if os.path.exists(lang_path):
            dest_file.write('\nfunction numeral_install(numeral) {\n')
            src = open(lang_path, 'r', 128*1024)
            dest_file.write(src.read().replace("""require('../numeral').""", "numeral.", 1))
            src.close()
            dest_file.write('\n}\n')
            return
        del lang_parts[-1]

def write_jquery_ui_datepicker_translation(js_path, locale, dest_file):
    lang = locale.replace("_", "-")
    path = False
    path1 = os.path.join(js_path, 'contrib', 'jquery-ui-1.10.4', 'i18n', 'jquery.ui.datepicker-%s.js' % lang)
    path2 = os.path.join(js_path, 'contrib', 'jquery-ui-1.10.4', 'i18n', 'jquery.ui.datepicker-%s.js' % lang[:2])
    for testpath in (path1, path2):
        if os.path.exists(testpath):
            path = testpath
            break
    if path:
        dest_file.write('\nfunction jQuery_ui_datepicker_install(jQuery) {\n')
        src = open(path, 'r', 128*1024)
        dest_file.write(src.read())
        src.close()
        dest_file.write('\n}\n')

def extract_xml(fileobj, keywords, comment_tags, options):
    """
    Extract translatable strings defined in manager XMLs
    """
    i18nattr = 'i18nattr'
    i18ntag = 'i18ntag'

    parser = et.XMLParser(remove_blank_text=True, remove_comments=False, remove_pis=True)
    root = safe_lxml.parse(fileobj, parser=parser)
    root = root.getroot()

    # sources format:  <t|a>:<tagname>[:attribute_name[=attribute_value]]
    if 'sources' in options:
        sources = [src.strip().split(':') for src in options['sources'].split(',')]
    else:
        return

    # make adhoc attribute translations by searching for tags with an attribute of i18nattr="attr1,attr2" (pointing on attributes)  and of i18ntag (pointing on tag value)
    adhocTranslnQuery = '//*[@%s]' % i18nattr
    for el in root.xpath(adhocTranslnQuery):
        i18nattr = el.attrib[i18nattr].split(',')
        for attr in i18nattr:
            attr = attr.strip()
            if attr in el.attrib:
                yield (el.sourceline, None, tuple([el.attrib[attr]]), ['@'+attr]) # (lineno, funcname, message, comments)

    adhocTranslnQuery = '//*[@%s]' % i18ntag
    for el in root.xpath(adhocTranslnQuery):
        yield (el.sourceline, None, tuple([el.text]), [el.tag]) # (lineno, funcname, message, comments)

    # execute configured extraction
    for src in sources:
        if len(src) == 2:
            src.append('')
        elif len(src) != 3:
            continue

        s = dict(zip(['type', 'tag', 'attrib'], src))

        if s['type'] == 't':
            attrib = s['attrib'].split('=')
            suffix = ''
            if len(attrib) == 1:
                if attrib[0] != '':
                    suffix = '[@%s]' % attrib[0]
            else:
                suffix = '[@%s="%s"]' % (attrib[0], attrib[1])

            xpathQuery = '//%s%s' % (s['tag'], suffix)
            for el in root.xpath(xpathQuery):
                yield (el.sourceline, None, tuple([el.text]), [s['tag']]) # (lineno, funcname, message, comments)

        elif s['type'] == 'a':
            xpathQuery = '//%s[@%s]' % (s['tag'], s['attrib'])
            for el in root.xpath(xpathQuery):
                attrib = el.attrib[s['attrib']]
                yield (el.sourceline, None, tuple([attrib]), ['@'+s['attrib']]) # (lineno, funcname, message, comments)


        elif s['type'] == 'e':
            attrib = s['attrib'].split('=')
            suffix = ''
            if len(attrib) == 1:
                if attrib[0] != '':
                    suffix = '[@%s]' % attrib[0]
            else:
                suffix = '[@%s="%s"]' % (attrib[0], attrib[1])

            xpathQuery = '//%s%s' % (s['tag'], suffix)
            for el in root.xpath(xpathQuery):
                text = el.text if el.text else ""
                text = text + ''.join(splunk.util.toDefaultStrings(et.tostring(c)) for c in el)
                yield (el.sourceline, None, tuple([text.strip()]), [s['tag']]) # (lineno, funcname, message, comments)

        else:
            continue

def extract_view(fileobj, keywords, comment_tags, options):
    if not hasattr(extract_view, 'viewExtractor'):
        extract_view.viewExtractor = ViewExtractor() # singleton
    root = safe_lxml.parse(fileobj)
    root = root.getroot()
    fileobj.seek(0)
    if root.tag == "view":#view
        return extract_view.viewExtractor.extract_view(fileobj)
    elif root.tag == 'form' or root.tag == 'dashboard': #form or dashboard
        return extract_xml(fileobj, keywords, comment_tags, options)
    return []

class ViewExtractor(object):
    """
    Helper singleton to extract translatable strings from view modules
    according to module's configuration
    """
    def __init__(self):
        import splunk.appserver.mrsparkle.lib.module as module
        self.modules = module.moduleMapper.getInstalledModules()

    def extract_view(self, fileobj):
        import splunk.appserver.mrsparkle.lib.viewconf as viewconf
        nativeObject = viewconf.loads(splunk.util.toDefaultStrings(fileobj.read()), fileobj.name)
        if 'label' in nativeObject:
            yield (0, None, nativeObject['label'], '')
        for moduleConfig in nativeObject['modules']:
            for entry in self.extract_module(moduleConfig):
                yield entry

    def extract_module(self, moduleConfig):
        moduleDefinition = self.modules.get("Splunk.Module." + moduleConfig['className'], {})
        if moduleDefinition:
            for param_name, param in moduleDefinition['params'].items():
                if 'translate' in param and param['translate'] and 'params' in moduleConfig and param_name in moduleConfig['params']:
                    translate = param['translate'].strip()
                    if translate == 'string':
                        yield (0, None, moduleConfig['params'][param_name], '')
                    elif translate:
                        for value in self.extract_params(translate, moduleConfig['params'][param_name]):
                            yield (0, None, value, '')
        if 'params' in moduleConfig:
            if 'group' in moduleConfig['params']:
                yield (0, None, moduleConfig['params']['group'], '')
            if 'altTitle' in moduleConfig['params']:
                yield (0, None, moduleConfig['params']['altTitle'], '')

        if 'children' in moduleConfig:
            for child in moduleConfig['children']:
                for entry in self.extract_module(child):
                    yield entry

    def extract_params(self, spec, param):
        if isinstance(spec, splunk.util.string_type):
            spec = spec.split(':')
        if spec and isinstance(param, (list, tuple)):
            if spec[0] == '*':
                for entry in param:
                    for v in self.extract_params(spec[1:], entry):
                        yield v
            else:
                try:
                    for v in self.extract_params(spec[1:], param[int(spec[0])]):
                        yield v
                except (ValueError, IndexError):
                    pass
        elif spec and isinstance(param, dict):
            if spec[0] == '*':
                for entry in param.values():
                    for v in  self.extract_params(spec[1:], entry):
                        yield v
            else:
                if spec[0] in param:
                    for v in  self.extract_params(spec[1:], param[spec[0]]):
                        yield v
        elif not spec and isinstance(param, splunk.util.string_type):
            if not str.isdigit(param): # don't translate numbers
                yield param

def extract_nav(fileobj, keywords, comment_tags, options):
    parser = et.XMLParser(remove_blank_text=True)
    nav = safe_lxml.fromstring(fileobj.read(), parser=parser)
    for node in nav.iter():
        if node.tag == 'a':
            yield (0, None, node.text, '')
        elif node.tag == 'collection':
            label = node.get('label')
            if label:
                yield (0, None, label, '')

def translate_view_params(spec, param):
    """
    Translate a view's module parameters according to the spec defined in the module's .conf file
    eg. "*:label" means translate all label fields in a list of dictionaries
    """
    if isinstance(spec, splunk.util.string_type):
        spec = spec.split(':')
    if spec and isinstance(param, (list, tuple)):
        if spec[0] == '*':
            for i in range(len(param)):
                param[i] = translate_view_params(spec[1:], param[i])
        else:
            try:
                param[int(spec[0])] = translate_view_params(spec[1:], param[int(spec[0])])
            except (ValueError, IndexError):
                pass
    elif spec and isinstance(param, dict):
        if spec[0] == '*':
            for key, entry in param.items():
                param[key] = translate_view_params(spec[1:], entry)
        else:
            if spec[0] in param:
                param[spec[0]] = translate_view_params(spec[1:], param[spec[0]])
    elif not spec and isinstance(param, splunk.util.string_type):
        if not str.isdigit(param): # don't translate numbers
            return(_(param))
        else:
            return param
    return param


def extract_conf(fileobj, keywords, comment_tags, options):
    """
    Extract translatable strings from conf files
    options is a dict:
    * extract_keys should be a comma separated list of key names to extract
    for times.conf this should be "label"
    """
    keys = options.get('extract_keys', '').split(',')
    prefix_keys = options.get('extract_prefix_keys', '').split(',')
    ln = 0
    lastval = []
    vcont = False
    keymatch = False
    for line in fileobj:
        line = splunk.util.toDefaultStrings(line)
        line = line.strip()
        ln += 1
        if vcont:
            lastval.append(line)
        elif '=' in line:
            key, val = line.split('=', 1)
            key = key.strip()
            val = val.strip()
            lastval = [ val ]
            keymatch = key in keys or [ True for pk in prefix_keys if key.startswith(pk) ]
        else:
            continue
        vcont = len(line) and line[-1] == '\\'
        if (not vcont) and keymatch:
            yield (ln, None, "\n".join(lastval), '') # (lineno, funcname, message, comments)

def extract_full_file(fileobj, keywords, comment_tags, options):
    yield (0, None, splunk.util.toDefaultStrings(fileobj.read()), '')

def extract_flat(fileobj, keywords, comment_tags, options):
    """
    Extract translatable strings defined in a flat text file

    Format:
    # TRANS: Comment for translator
    String to be translated

    There must be a blank line between each separate translation.
    Plural strings can be specified by following the singular form with another string:

    # TRANS: This is shown when the user hits the delete key
    Delete %d file?
    Delete %d files?
    """
    ln = 0
    comments = []
    messages = []
    for line in fileobj:
        line = splunk.util.toDefaultStrings(line)
        ln += 1
        line = line.strip()
        if not len(line):
            if messages:
                funcname = 'ungettext' if len(messages)>1 else '_'
                yield (ln, funcname, tuple(messages), comments) # (lineno, funcname, message, comments)
                messages = []
                comments = []
            continue
        if line[0] == '#':
            comment = line[1:].strip()
            for tag in comment_tags:
                if comment.startswith(tag):
                    comments.append(comment)
        else:
            messages.append(line)
    if messages:
        yield (ln, None, messages, comments) # (lineno, funcname, message, comments)

class DebugDateTimePattern(babel.dates.DateTimePattern):
    def apply(self, dt, locale):
        return super(DebugDateTimePattern, self).apply(datetime(3333, 11, 22, 11, 22, 33, 123000), locale)

class DebugLocale(babel.Locale):
    def __init__(self, *a, **kw):
        super(DebugLocale, self).__init__(*a, **kw)
        dummy = self._data['months'] # load the db
        self._Locale__data['months'] = self.__recursive_set_dict_strings(self._Locale__data['months'], 'M')
        self._Locale__data['days'] = self.__recursive_set_dict_strings(self._Locale__data['days'], 'D')
        self._Locale__data['day_periods'] = self.__recursive_set_dict_strings(self._Locale__data['day_periods'], 'X')
        self._Locale__data['date_formats'] = dict([ (k, DebugDateTimePattern(v.pattern, v.format)) for (k, v) in self._Locale__data['date_formats'].items() ])
        self._Locale__data['time_formats'] = dict([ (k, DebugDateTimePattern(v.pattern, v.format)) for (k, v) in self._Locale__data['time_formats'].items() ])

    def __recursive_set_dict_strings(self, d, ch):
        """
        Change all string values to a repeating character for easy identification
        """
        result = {}
        for k, v in d.items():
            if isinstance(v, dict) or isinstance(v, babel.localedata.LocaleDataDict):
                result[k] = self.__recursive_set_dict_strings(v, ch)
            elif isinstance(v, (tuple, list)):
                result[k] = [ ch*len(x) for x in v ]
            elif isinstance(v, splunk.util.string_type):
                result[k] = ch*len(v)
            else:
                result[k] = v
        return result


# Use our request/thread-aware version of ugettext as the global _ shortcut
if sys.version_info >= (3, 0):
    import builtins as builtins
else:
    import __builtin__ as builtins
builtins.__dict__['_'] = ugettext
builtins.__dict__['ungettext'] = ungettext
