import _ from 'underscore';
import util from 'splunk.util';
import DashboardFactory from 'dashboard/DashboardFactory';
import FormManager from 'dashboard/manager/FormManager';
import tokenUtils from 'splunkjs/mvc/tokenutils';
import {
    getDrilldownPropertyValue,
    getEnabledDrilldownAttribute,
    getDisabledDrilldownAttribute,
} from 'controllers/dashboard/helpers/ReportModelHelper';
import DashboardElementReport from 'models/dashboard/DashboardElementReport';
import {
    EXPLICIT_OPTION,
    TOKEN_OPTION,
    GLOBAL_OPTION,
} from 'dashboard/containers/editor/drilldown/search/timeRangeOptionNames';
import {
    NO_DRILLDOWN,
    LINK_TO_SEARCH,
    LINK_TO_DASHBOARD,
    LINK_TO_REPORT,
    LINK_TO_CUSTOM_URL,
    EDIT_TOKENS,
} from 'dashboard/containers/editor/drilldown/drilldownNames';
import * as DrilldownTokens from 'controllers/dashboard/helpers/DrilldownTokens';

const isSupported = (eventManager) => {
    try {
        const events = eventManager.settings.get('events');
        const drilldown = events.find(event => event.type === 'drilldown');
        const conditions = drilldown.settings.conditions || [];
        const linkActions = _.filter(drilldown.settings.actions,
            action => action.type === 'link',
        );
        const tokenActions = _.filter(drilldown.settings.actions,
            action => action.type === 'set' || action.type === 'unset' || action.type === 'eval',
        );

        const hasNoCondition = conditions.length === 0;
        const isValidDefault = linkActions.length === 0 && tokenActions.length === 0;
        const isValidLink = linkActions.length === 1 && tokenActions.length === 0;
        const isValidSetUnsetEval = linkActions.length === 0 && tokenActions.length >= 1;

        return hasNoCondition && (isValidDefault || isValidLink || isValidSetUnsetEval);
    } catch (err) {
        return true;
    }
};

const isCustomViz = reportModel => DashboardElementReport.getVizType(reportModel) === 'viz';

const getDrilldownDefinition = (eventManager) => {
    if (!eventManager) {
        return null;
    }

    const events = eventManager.settings.get('events');
    return events.find(event => event.type === 'drilldown');
};

const isDrilldownTurnedOn = (report) => {
    const disabledOption = getDisabledDrilldownAttribute(report.entry.content.toJSON());

    /*
    for <event>, the option looks like this (multiple entries): {
        'display.list.drilldown': none,
        'display.raw.drilldown': none,
        'display.table.drilldown': 0,
    }
    for others, the option looks like this (one entry): {
        'display.statistics.drilldown': none
    }
     */
    return _.find(disabledOption, (value, key) => report.entry.content.get(key) !== value);
};

const turnOffDrilldown = report =>
    report.entry.content.set(getDisabledDrilldownAttribute(report.entry.content.toJSON()));

const turnOnDrilldown = report =>
    report.entry.content.set(getEnabledDrilldownAttribute(report.entry.content.toJSON()));

const getElementSearchTimeRange = (managerReportContent, searchModel) => {
    // The purpose of this function is to resolve SPL-133691, which requires the default time range
    // option to be the same in both Edit Search dialog and Edit Drilldown dialog.
    // So this function mimics the logic (but not exactly the same) in Edit Search dialog to decide
    // what would be the default time range option.
    // The reason we need 'searchModel' is because 'searchModel' is generated by the EditSearch.js dialog,
    // user may open drilldown editor immediately after saving the EditSearch dialog.
    // eslint-disable-next-line max-len
    const earliest = searchModel ? searchModel.get('earliest_time') : managerReportContent.get('dispatch.earliest_time');
    const latest = searchModel ? searchModel.get('latest_time') : managerReportContent.get('dispatch.latest_time');

    let tokenOption = EXPLICIT_OPTION;

    if ((_.isUndefined(earliest) && _.isUndefined(latest)) ||
        (earliest === '$earliest$' && latest === '$latest$')) {
        tokenOption = GLOBAL_OPTION;
    } else if (tokenUtils.isToken(earliest) && tokenUtils.isToken(latest)) {
        tokenOption = TOKEN_OPTION;
    }

    const timeRangeSettings = {
        activeTimeRangeOption: tokenOption,
    };

    if (tokenOption === EXPLICIT_OPTION) {
        return Object.assign({}, timeRangeSettings, {
            activeTimeRange: {
                earliest,
                latest,
            },
        });
    }

    return Object.assign({}, timeRangeSettings, {
        activeTimeRangeToken: {
            earliest: tokenUtils.getTokenName(earliest),
            latest: tokenUtils.getTokenName(latest),
        },
    });
};

const getToSearch = ({ report, managerReportContent, searchModel }, eventManager) => { // eslint-disable-line max-len
    const timeItems = FormManager.getInputs().filter(input => (
        input.settings.get('type') === 'time'
    )).map(timeInput => ({
        label: util.sprintf(_('Shared Time Picker (%s)').t(), timeInput.settings.get('token')),
        value: timeInput.settings.get('token'),
    }));

    const elementSearchTimeRange = getElementSearchTimeRange(managerReportContent, searchModel);

    const toSearchDefault = Object.assign({
        options: [
            {
                label: _('Auto').t(),
                value: 'default',
            },
            {
                label: _('Custom').t(),
                value: 'custom',
            },
        ],
        activeOption: 'default',
        search: report.entry.content.get('search'),
        searchError: '',
        target: '_blank',
        extraTimeRangeOptions: timeItems,
        activeTimeRange: {},
        activeTimeRangeToken: {},
        earliestTokenError: '',
        latestTokenError: '',
    }, elementSearchTimeRange);

    try {
        const drilldown = getDrilldownDefinition(eventManager);

        const linkAction = (drilldown.settings.actions).find(
            action => action.type === 'link',
        );

        // possible string 1) 'search?q=index=_audit&earliest=-15m&latest=now'
        // note: need to match line terminators as well.
        const searchWithTimeRange = linkAction.value.match(/^search\?q=([\s\S]*)&earliest=([\s\S]*)&latest=([\s\S]*)/);
        // possible string 2) 'search?q=index=_audit'
        const onlySearchString = linkAction.value.match(/^search\?q=(.*)/);
        let search;

        if (searchWithTimeRange) {
            try {
                search = decodeURIComponent(searchWithTimeRange[1]);
            } catch (err) {
                search = searchWithTimeRange[1];
            }
            const earliest = searchWithTimeRange[2];
            const latest = searchWithTimeRange[3];
            let activeTimeRangeOption;
            let activeTimeRangeToken;
            let activeTimeRange;

            if (tokenUtils.isToken(earliest) && tokenUtils.isToken(latest)) {
                const earliestToken = tokenUtils.getTokenName(earliest);
                const latestToken = tokenUtils.getTokenName(latest);
                const earliestTokenPrefix = earliestToken.replace(/\.?earliest$/g, '');
                const latestTokenPrefix = latestToken.replace(/\.?latest$/g, '');
                if (earliestTokenPrefix === latestTokenPrefix
                    && timeItems.find(item => (item.value === earliestTokenPrefix))) {
                    // use the time token name as option
                    activeTimeRangeOption = earliestTokenPrefix;
                } else {
                    if (earliestToken === 'earliest' && latestToken === 'latest') {
                        activeTimeRangeOption = GLOBAL_OPTION;
                    } else {
                        activeTimeRangeOption = TOKEN_OPTION;
                    }
                    activeTimeRangeToken = {
                        earliest: earliestToken,
                        latest: latestToken,
                    };
                }
            } else {
                activeTimeRangeOption = EXPLICIT_OPTION;
                activeTimeRange = {
                    earliest,
                    latest,
                };
            }
            return _.defaults({}, {
                activeOption: 'custom',
                target: linkAction.target || '_self',
                search,
                activeTimeRangeOption,
                activeTimeRangeToken,
                activeTimeRange,
            }, toSearchDefault);
        } else if (onlySearchString) {
            try {
                search = decodeURIComponent(onlySearchString[1]);
            } catch (err) {
                search = onlySearchString[1];
            }
            return _.defaults({}, {
                activeOption: 'custom',
                target: linkAction.target || '_self',
                search,
                activeTimeRangeOption: EXPLICIT_OPTION,
                activeTimeRange: {
                    earliest: '',
                    latest: '',
                },
            }, toSearchDefault);
        }
    } catch (err) {
        return toSearchDefault;
    }

    return toSearchDefault;
};

const hasAppName = link => /^\/app\/[^/]+\/\w+/.test(link);
const getAppName = (link) => {
    const matches = link.match(/^\/app\/([^/]+)\/\w+/);
    if (matches) {
        return matches[1];
    }

    return undefined;
};

// note: '/app/<app_name>/search?q=blabla' will be treated as a splunk page url, because currently the drilldown editor
// does not support specifying the app name when "Link to Custom Search", so the url is not a valid "Link to Custom
// Search" url.
const isSearchLink = link => /^search\?q=.*/.test(link);
const isReportLink = link => /^(\/app\/[^/]+\/)?report\?s=.*/.test(link);
const isCustomURL = link => /^https?:\/\//.test(link);
const isDashboardLink = link => !isSearchLink(link) && !isReportLink(link) && !isCustomURL(link);

const getReportName = (link) => {
    let reportId;

    if (hasAppName(link)) {
        reportId = link.match(/^\/app\/[^/]+\/report\?s=(.+)/)[1];    // example: '/app/search/report?s=report_name'
    } else {
        reportId = link.slice(9);  // example: 'report?s=report_name'
    }

    const decodedReportId = decodeURIComponent(reportId);

    const fullPathPattern = /^\/servicesNS\/[^/]+\/[^/]+\/saved\/searches\/(.*)/;

    const match = decodedReportId.match(fullPathPattern);

    let reportName;

    if (match) {
        reportName = match[1];
    } else {
        reportName = decodedReportId;
    }

    // need decode report name again
    try {
        reportName = decodeURIComponent(reportName);
    } catch (err) {
        /* continue regardless of error */
    }
    return reportName;
};

const getDashboardNameAndParams = (link) => {
    if (hasAppName(link)) {
        return link.match(/^\/app\/[^/]+\/(.*)/)[1];
    }

    return link;
};

const getDashboardName = (link) => {
    const nameAndParams = getDashboardNameAndParams(link);

    return nameAndParams.replace(/\?.*/, '');
};
const getDashboardParams = (link) => {
    const nameAndParams = getDashboardNameAndParams(link);

    const queryString = nameAndParams.split('?')[1];

    if (!queryString) {
        return [];
    }

    const props = util.queryStringToProp(queryString);

    return _.map(props, (value, key) => ({
        key,
        value,
    }));
};

const getToDashboard = (model, eventManager) => {
    const defaultToDashboard = {
        activeApp: model.application.get('app'),
        activeDashboard: '',
        // have a placeholder so that by default an key-value pair input shows up in the advanced
        // section.
        params: [{ key: '', value: '' }],
        target: '_blank',
    };

    try {
        const drilldown = getDrilldownDefinition(eventManager);
        const linkAction = (drilldown.settings.actions).find(
            action => action.type === 'link',
        );
        if (!isDashboardLink(linkAction.value)) {
            return defaultToDashboard;
        }

        const params = getDashboardParams(linkAction.value);

        return _.defaults({}, {
            activeApp: getAppName(linkAction.value),
            target: linkAction.target || '_self',   // unspecified target means open in current tab.
            activeDashboard: getDashboardName(linkAction.value),
            // set params to undefined if the array is empty, so that defaultToDashboard could
            // override it.
            params: _.isEmpty(params) ? undefined : params,
        }, defaultToDashboard);
    } catch (err) {
        return defaultToDashboard;
    }
};
const isToDashboard = state => !!state.activeDashboard;

const getToReport = (model, eventManager) => {
    const toReportDefault = {
        activeApp: model.application.get('app'),
        activeReport: '',
        target: '_blank',
    };

    try {
        const drilldown = getDrilldownDefinition(eventManager);
        const linkAction = (drilldown.settings.actions).find(
            action => action.type === 'link',
        );
        if (!isReportLink(linkAction.value)) {
            // Not a valid report link
            return toReportDefault;
        }

        return _.defaults({}, {
            activeApp: getAppName(linkAction.value),
            target: linkAction.target || '_self',
            activeReport: getReportName(linkAction.value),
        }, toReportDefault);
    } catch (err) {
        return toReportDefault;
    }
};
const isToReport = state => !!state.activeReport;

const getToURL = (eventManager) => {
    const toURLDefault = {
        target: '_blank',
    };

    try {
        const drilldown = getDrilldownDefinition(eventManager);
        const linkAction = (drilldown.settings.actions).find(
            action => action.type === 'link',
        );
        try {
            linkAction.value = decodeURI(linkAction.value);
        } catch (err) {
            /* continue regardless of error */
        }
        return _.defaults({}, {
            url: linkAction.value,
            target: linkAction.target,
        }, toURLDefault);
    } catch (err) {
        return toURLDefault;
    }
};
const isToURL = state => isCustomURL(state.url);

const getTokens = (eventManager) => {
    /*
     sample:
     [
     {
     "type": "set",
     "token": "foo",
     "value": "$row.count$"
     },
     {
     "type": "unset",
     "token": "bar"
     },
     {
     "type": "eval",
     "token": "foo",
     "value": "bar",
     }
     ]
     */

    // this is to give user an empty token input box so that user can start typing token names and
    // values without clicking the "add new" button.
    const defaultTokens = [{
        type: 'set',
        token: '',
        value: '',
    }];

    try {
        const drilldown = getDrilldownDefinition(eventManager);
        const tokens = drilldown.settings.actions.filter(action =>
            action.type === 'set' || action.type === 'eval' || action.type === 'unset',
        );

        return { items: tokens.length > 0 ? tokens : defaultTokens };
    } catch (err) {
        return { items: defaultTokens };
    }
};

const isTokens = state => state.items.filter(token => token.type && token.token).length > 0;

const getActiveAction = (model, eventManager) => {
    const drilldownPropertyValue = getDrilldownPropertyValue(model.report.entry.content.toJSON());
    if (tokenUtils.isToken(drilldownPropertyValue)) {
        return drilldownPropertyValue;
    } else if (!isDrilldownTurnedOn(model.report)) {
        return NO_DRILLDOWN;
    } else if (isToReport(getToReport(model, eventManager))) {
        // NOTE the detection of report should be before the detection of dashboard, because report
        // is a special case.
        return LINK_TO_REPORT;
    } else if (isToDashboard(getToDashboard(model, eventManager))) {
        return LINK_TO_DASHBOARD;
    } else if (isToURL(getToURL(eventManager))) {
        return LINK_TO_CUSTOM_URL;
    } else if (isTokens(getTokens(eventManager))) {
        return EDIT_TOKENS;
    }

    // NOTE: this condition must be the last one because there's case that there's no valid
    // definition of drilldown so that it falls back on default drilldown behavior which is
    // link to search.
    return LINK_TO_SEARCH;
};

/**
 * this function is to provide the initial data for drilldown editor store. The initial data is
 * converted from reportModel and eventManager
 * @param model {object} this.model, should contain report
 * @param eventManager {object} EventManager that contains drilldown actions.
 */
const createInitialState = ({ model, eventManager }) => ({
    isSupported: isSupported(eventManager),
    isCustomViz: isCustomViz(model.report),
    activeAction: getActiveAction(model, eventManager),
    forms: {
        [LINK_TO_SEARCH]: getToSearch(model, eventManager),
        [LINK_TO_DASHBOARD]: getToDashboard(model, eventManager),
        [LINK_TO_REPORT]: getToReport(model, eventManager),
        [LINK_TO_CUSTOM_URL]: getToURL(eventManager),
        [EDIT_TOKENS]: getTokens(eventManager),
    },
    elementEnv: {
        // elementEnv contains information of dashboard element, such as viz, search. THis info is
        // ready-only.
        drilldownTokens: DrilldownTokens.getTokens(model.report),
    },
    // splunkEnv contains environmental information which are unlikely to change, this information
    // is read-only.
    splunkEnv: {
        application: model.application.toJSON(),
    },
});

/*
code above is for: SimpleXML -> redux state
=============================================
code below is for: redux state -> SimpleXML
 */

/**
 * this is to generate url for "Link to dashboard" action.
 * @param state {object} store.getState()
 * @param absolute {boolean} whether the url should be absolute. If true, url looks like this:
 *      /app/<app_name>/<dashboard_name>. If false AND the target app is the same as current app,
 *      url looks like this: <dashboard_name>.
 * @returns {string} url that links to a dashboard.
 */
const generateLinkToDashboard = (state, { absolute } = {}) => {
    const { activeApp, activeDashboard, params } = state.forms[LINK_TO_DASHBOARD];

    if (!activeDashboard) {
        return '';
    }

    const queryString = params.filter(({ key }) => !!key).map(
        ({ key, value }) => `${key}=${value}`).join('&');

    const nameAndParams = queryString ? `${activeDashboard}?${queryString}` : activeDashboard;

    // if target app is the same as current app and 'absolute' option is false, then there's no need
    // to add /app/<app_name> prefix.
    if (activeApp === state.splunkEnv.application.app && absolute === false) {
        return nameAndParams;
    }

    return `/app/${activeApp}/${nameAndParams}`;
};

const getEventManagerSetting = (state) => {
    const drilldown = {
        type: 'drilldown',
    };
    let setting;
    let earliest;
    let latest;
    switch (state.activeAction) {
        case LINK_TO_SEARCH:
            setting = state.forms[LINK_TO_SEARCH];
            switch (setting.activeTimeRangeOption) {
                case TOKEN_OPTION:
                    earliest = `$${setting.activeTimeRangeToken.earliest}$`;
                    latest = `$${setting.activeTimeRangeToken.latest}$`;
                    break;
                case EXPLICIT_OPTION:
                    earliest = setting.activeTimeRange.earliest;
                    latest = setting.activeTimeRange.latest;
                    break;
                case GLOBAL_OPTION:
                    earliest = '$earliest$';
                    latest = '$latest$';
                    break;
                default:
                    // binds to time input
                    // activeTimeRangeOption is the time token name in this case.
                    earliest = `$${setting.activeTimeRangeOption}.earliest$`;
                    latest = `$${setting.activeTimeRangeOption}.latest$`;
            }
            try {
                let search = setting.search;
                const hasToken = tokenUtils.hasToken(search);
                search = encodeURIComponent(search);
                if (hasToken) {
                    // replace encoded $ back to $
                    search = search.replace(/%24/g, '$');
                    // replace encoded token filters |s, |h, |u, |n back
                    search = search.replace(/%7C(?=s|u|h|n)/g, '|');
                }
                setting.search = search;
            } catch (err) {
                /* continue regardless of error */
            }
            Object.assign(drilldown, {
                settings: {
                    actions: [{
                        target: setting.target,
                        type: 'link',
                        value: `search?q=${setting.search}&earliest=${earliest}&latest=${latest}`,
                    }],
                },
            });
            break;
        case LINK_TO_DASHBOARD:
            setting = state.forms[LINK_TO_DASHBOARD];
            Object.assign(drilldown, {
                settings: {
                    actions: [{
                        target: setting.target,
                        type: 'link',
                        value: encodeURI(generateLinkToDashboard(state)),
                    }],
                },
            });
            break;
        case LINK_TO_REPORT:
            setting = state.forms[LINK_TO_REPORT];
            Object.assign(drilldown, {
                settings: {
                    actions: [{
                        target: setting.target,
                        type: 'link',
                        value: encodeURI(`/app/${setting.activeApp}/report?s=${setting.activeReport}`),
                    }],
                },
            });
            break;
        case LINK_TO_CUSTOM_URL:
            setting = state.forms[LINK_TO_CUSTOM_URL];
            Object.assign(drilldown, {
                settings: {
                    actions: [{
                        target: setting.target,
                        type: 'link',
                        value: setting.url,
                    }],
                },
            });
            break;
        case EDIT_TOKENS:
            setting = state.forms[EDIT_TOKENS];
            Object.assign(drilldown, {
                settings: {
                    // filter out invalid tokens
                    actions: _.filter(setting.items, token => token.type && token.token),
                },
            });
            break;
        default:
            break;
    }
    return {
        type: 'event-manager',
        settings: {
            events: [drilldown],
        },
    };
};

const removeEventManager = (settings, model, eventManager) => {
    settings.unset('evtmanagerid');
    model.report.entry.content.unset('dashboard.element.evtmanagerid');
    if (eventManager) {
        eventManager.dispose();
    }
    model.controller.trigger('edit:delete-drilldown', {
        eventManagerId: eventManager ? eventManager.id : null,
        elementId: settings.get('id'),
    });
};

/**
 * this converts redux store back to setting, model and eventManager.
 * @param state {object} store.getState()
 * @param settings {object} dashboard element settings.
 * @param model {object} this.model, should contain report.
 * @param eventManager {object} (optional) eventManager that contains the drilldown actions. If no
 *  event manager is defined, it will create one.
 */
const applyState = ({ state, settings, model, eventManager }) => {
    if (state.activeAction === NO_DRILLDOWN) {
        // clean up event manager
        turnOffDrilldown(model.report);
        removeEventManager(settings, model, eventManager);
    } else {
        // SPL-130624: this condition is needed, otherwise when user click save it may automatically
        // change the drilldown option from "Row" to "Cell" if viz type is table.
        if (!isDrilldownTurnedOn(model.report)) {
            turnOnDrilldown(model.report);
        }
        // auto drilldown, no event manager is required
        if (state.activeAction === LINK_TO_SEARCH &&
            state.forms[LINK_TO_SEARCH].activeOption === 'default') {
            removeEventManager(settings, model, eventManager);
            return;
        }
        const setting = getEventManagerSetting(state);
        const options = {};
        let manager = null;
        if (eventManager) {
            manager = eventManager;
            manager.applySetting(setting.settings);
        } else {
            manager = DashboardFactory.getDefault().instantiate(setting, options);
            settings.set('evtmanagerid', manager.id);
        }
        model.controller.trigger('edit:drilldown', {
            eventManagerId: manager.id,
            eventManager: manager,
            elementId: settings.get('id'),
        });
    }
};

export {
    createInitialState,
    generateLinkToDashboard,
    applyState,
    // the functions below are exported only for testing purpose, they are considered private functions
    // that should only be used in this module.
    getReportName,
};
